/* =============================================================================
 *
 * utils.c
 *
 * This file contains the following functions.
 *      scroll()
 *      print_welcome()
 *      show_demo_title()
 *      report_error()
 *
 * v1.0		YFS		Initial release.
 *
 * =============================================================================
*/

/*
* Copyright (2014), Cypress Semiconductor Corporation. All Rights Reserved.
*
* This software is owned by Cypress Semiconductor Corporation (Cypress)
* and is protected by and subject to worldwide patent protection (United
* States and foreign), United States copyright laws and international treaty
* provisions. Cypress hereby grants to licensee a personal, non-exclusive,
* non-transferable license to copy, use, modify, create derivative works of,
* and compile the Cypress Source Code and derivative works for the sole
* purpose of creating custom software in support of licensee product to be
* used only in conjunction with a Cypress integrated circuit as specified in
* the applicable agreement. Any reproduction, modification, translation,
* compilation, or representation of this software except as specified above 
* is prohibited without the express written permission of Cypress.
*
* Disclaimer: CYPRESS MAKES NO WARRANTY OF ANY KIND, EXPRESS OR IMPLIED, WITH 
* REGARD TO THIS MATERIAL, INCLUDING, BUT NOT LIMITED TO, THE IMPLIED 
* WARRANTIES OF MERCHANTABILITY AND FITNESS FOR A PARTICULAR PURPOSE.
* Cypress reserves the right to make changes without further notice to the 
* materials described herein. Cypress does not assume any liability arising out 
* of the application or use of any product or circuit described herein. Cypress 
* does not authorize its products for use as critical components in life-support 
* systems where a malfunction or failure may reasonably be expected to result in 
* significant injury to the user. The inclusion of Cypress' product in a life-
* support systems application implies that the manufacturer assumes all risk of 
* such use and in doing so indemnifies Cypress against all charges. 
*
* Use of this Software may be limited by and subject to the applicable Cypress
* software license agreement. 
*/

/* Include FreeRTOS APIs and defines */
#include <FreeRTOS.h>

/* Include PSoC system and component APIs and defines */
#include <project.h>

/* Include application function declarations and defines */
#include <utils.h>
#include <App_Defs.h>

/*
 * Function:	scroll
 *
 * Slide text across the LCD (right-to-left). Abort early if SW2 pressed.
 *
 * Globals:		str uses a static buffer to save stack space (not reentrant)
 *
 * Parameters:	row (0 or 1)
 *				msg to display
 *
 * Return:		None
 */
void scroll( int row, char *msg )
{
	int i, j;						/* General purpose loop counters */
	static char str[MAX_MSG_LEN];	/* Local buffer for the string */
	int len = 0;					/* Calculated length of the string */
	
	/* Clear (spaces) the first row of text */
	for( i=0; i<LCD_WIDTH; i++ )
	{
		str[len++] = ' ';
	}
	
	/* Append the message */
	while( *msg )
	{
		str[len++] = *msg++;
		
		/* Just ignore strings that are too long */
		if( len > MAX_MSG_LEN )
			return;
	}
	
	/* Terminate the new string */
	str[len] = '\0';
	
	/*
	Scroll the text across the screen on the given row. In each loop LCD_WIDTH
	chars are printed. Then the start point is moved by char and the next
	LCD_WIDTH are printed to create a right-left motion. At the end of the
	string (null char) a space is printed (to clean up the character printed
	in the previous loop) and the inner loop exited.
	*/
	for( i=0; i<len; i++ )
	{
		LCD_Position( row, ROW_START );
		
		for(j=0; j<16; j++ )
		{
			/* Print LCD_WIDTH characters */
			if( str[i+j] )
			{
				LCD_PutChar( str[i+j] );
			}
			else
			{
				/* End of the string - just print a space and exit loop */
				LCD_PutChar( ' ' );
				break;
			}
			
			/* Check SW2 for an abort of the scrolling */
			if( ! P6_1_Read() )
			{
				/* Momentarily report the rest of the info is being skipped */
				LCD_Position( row, ROW_START );
				LCD_PrintString( "Skipping...     " );
			
				CyDelay( 500 );		/* 500ms */
			
				return;
			} /* if( ! P6_1_Read() ) */
			
		} /* for(j=0; j<16; j++ ) */
		
		/* Non-OS delay call to slow the scroll rate - 250ms */
		CyDelay( 250 );
		
	} /* for( i=0; i<len; i++ ) */
	
} /* scroll() */

/*
 * Function:	print_welcome
 *
 * Print the welcome text on the LCD.
 *
 * Globals:		None
 *
 * Parameters:	None
 *
 * Return:		None
 */
void print_welcome( void )
{
	/* Print the program title "FreeRTOS x PSoC" on the top row */
	LCD_ClearDisplay();
	LCD_PrintString( "FreeRTOS " );
	LCD_PutChar( CHAR_HEART );
	LCD_PrintString( " PSoC" );
} /* print_welcome() */

/*
 * Function:	show_demo_title
 *
 * Print the demo title on the LCD.
 *
 * Globals:		None
 *
 * Parameters:	index - the number of the demo to show
 *				str - the demo name
 *
 * Return:		None
 */
void show_demo_title( int index, char *str )
{
	/* Clear the row */
	LCD_Position( BOTTOM_ROW, ROW_START );
	LCD_PrintString( SIXTEENSPACES );
	
	/* Adjust for the human tendency to count from 1, not 0 */
	index++;
	
	/* Print the demo number and name */
	LCD_Position( BOTTOM_ROW, ROW_START );
	LCD_PrintNumber( index );
	LCD_PrintString( ": " );
	LCD_PrintString( str );
} /* show_demo_title() */

/*
 * Function:	report_error
 *
 * Print an error condition on the LCD.
 *
 * Globals:		None
 *
 * Parameters:	str - the error message
 *
 * Return:		Never returns
 */
void report_error( char *str )
{
    int index = 0;
    
    /* Clear the LCD and print Error! */
    LCD_ClearDisplay();
	LCD_PrintString( "Error!" );
    
    /* Print the message, up to LCD_WIDTH chars */
    LCD_Position( BOTTOM_ROW, ROW_START );    
    while( index < LCD_WIDTH && str[index] != '\0' )
    {
        LCD_PutChar( str[index++] );
    }
    
    while( 1 )
    {
        /* Never return */
    }
    
} /* report_error() */
